# 機能設計書 2-数学演算（Math Operations）

## 概要

本ドキュメントは、TensorFlowのコアフレームワークにおける数学演算（Math Operations）機能の設計を記述する。加減乗除・行列演算・統計関数・三角関数などの数学的演算を提供する。

### 本機能の処理概要

数学演算機能は、テンソルに対する基本的な算術演算から高度な数学関数まで幅広い数値計算を提供する。要素ごとの演算、リダクション（集約）演算、セグメント演算、行列積演算など、機械学習に必要なすべての数学的操作を網羅する。

**業務上の目的・背景**：機械学習モデルの訓練と推論は本質的に数学的演算の連鎖である。損失関数の計算、勾配の算出、正規化処理、特徴量のスケーリングなど、あらゆる場面で数学演算が必要となる。本機能はこれらの演算をGPU/TPU上で高速に実行できる形で提供し、大規模な機械学習ワークロードを支える。

**機能の利用シーン**：(1) 損失関数の計算（reduce_mean, log, exp）、(2) 正規化処理（reduce_mean, reduce_sum）、(3) 行列演算（matmul）、(4) 比較演算（equal, less, greater）、(5) 統計量計算（reduce_min, reduce_max, argmax）、(6) 累積演算（cumsum, cumprod）。

**主要な処理内容**：
1. 算術演算: add, subtract, multiply, divide, pow, abs, square, sqrt
2. 三角関数: sin, cos, tan, asin, acos, atan
3. 指数・対数: exp, log, log1p, expm1, sigmoid
4. リダクション: reduce_sum, reduce_mean, reduce_max, reduce_min, reduce_prod
5. 行列演算: matmul（行列積）
6. 比較演算: equal, not_equal, less, greater, less_equal, greater_equal
7. セグメント演算: segment_sum, unsorted_segment_sum
8. 累積演算: cumsum, cumprod

**関連システム・外部連携**：C++のgen_math_opsカーネルを通じてCPU/GPU上で実行される。MKL（Intel Math Kernel Library）との統合による最適化も含まれる。NumPyスタイルのブロードキャスト規則をサポートする。

**権限による制御**：特になし。すべてのユーザが利用可能である。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はコアAPIのため直接的な画面関連はなし |

## 機能種別

計算処理（数学演算処理）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| x / input_tensor | tf.Tensor | Yes | 演算対象テンソル | 数値型テンソルであること |
| y | tf.Tensor | 二項演算時 | 第2オペランド | xとブロードキャスト互換であること |
| axis | int / list[int] | リダクション時 | リダクション対象軸 | テンソルのランク範囲内 |
| keepdims | bool | No | リダクション後の次元保持 | - |
| name | string | No | 操作名 | - |

### 入力データソース

Python API呼び出し時の引数として入力される。tf.convert_to_tensorにより暗黙的にPythonスカラーやNumPy配列もテンソルに変換される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result | tf.Tensor | 演算結果テンソル |

### 出力先

呼び出し元のPythonコードに返却される。

## 処理フロー

### 処理シーケンス

```
1. Python API呼び出し受付
   └─ @tf_export / @dispatch.add_dispatch_support デコレータ処理
2. 入力テンソル変換
   └─ ops.convert_to_tensor_v2_with_dispatch()
3. ブロードキャスト形状計算（二項演算時）
   └─ shape_inference::BroadcastBinaryOpShapeFn
4. gen_math_ops経由でC++カーネル呼び出し
   └─ Eager: 即時実行 / Graph: ノード追加
5. 結果テンソル返却
```

### フローチャート

```mermaid
flowchart TD
    A[Python API呼び出し] --> B[入力テンソル変換]
    B --> C{二項演算か?}
    C -->|Yes| D[ブロードキャスト形状チェック]
    C -->|No| E[単項演算実行]
    D --> F[gen_math_ops呼び出し]
    E --> F
    F --> G{リダクション演算か?}
    G -->|Yes| H[axis方向に集約]
    G -->|No| I[要素ごと演算]
    H --> J[結果テンソル返却]
    I --> J
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | ブロードキャスト | 二項演算でNumPyスタイルのブロードキャスト規則を適用 | 二項演算時 |
| BR-02 | 型昇格 | 異なる数値型間の演算時に型昇格を適用 | 異なるdtype混在時 |
| BR-03 | NaN伝播 | NaNを含む演算はNaNを伝播させる | 浮動小数点演算時 |
| BR-04 | ゼロ除算 | 整数ゼロ除算はエラー、浮動小数点はinf/NaN | divide操作時 |

### 計算ロジック

- reduce_mean: sum(input, axis) / count(axis)
- matmul: 標準行列積 C[i,j] = sum_k(A[i,k] * B[k,j])
- sigmoid: 1 / (1 + exp(-x))
- cumsum: output[i] = sum(input[0:i+1])

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作は発生しない |

### テーブル別操作詳細

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| InvalidArgumentError | 形状エラー | matmulの内側次元が不一致 | 行列の形状を確認する |
| InvalidArgumentError | 型エラー | 非数値型テンソルでの数学演算 | 数値型テンソルを使用する |
| InvalidArgumentError | ブロードキャストエラー | ブロードキャスト不可能な形状 | 入力形状を揃える |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。数学演算はステートレスな計算である。

## パフォーマンス要件

- 要素ごと演算はGPU上でデータ並列に実行されること
- matmul操作はcuBLAS/MKLカーネルを活用し最適化されること
- リダクション演算はツリー構造の並列削減を使用すること

## セキュリティ考慮事項

- 数値オーバーフロー/アンダーフローに対する適切なハンドリング
- NaN/Inf値の意図しない伝播の防止

## 備考

- Python演算子（+, -, *, /等）のオーバーロードがtensor_math_operator_overrides.pyで定義されている
- MKL最適化バージョンのopが一部存在する（math_ops.ccのコメント参照）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

数学演算で扱うテンソルのデータ型と演算子オーバーロードの仕組みを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | dtypes.py | `tensorflow/python/framework/dtypes.py` | 数値型（float32, float64, int32等）の定義 |
| 1-2 | tensor_math_operator_overrides.py | `tensorflow/python/ops/tensor_math_operator_overrides.py` | Python演算子（+,-,*,/）のオーバーロード |

**読解のコツ**: TensorFlowの数学演算は `override_binary_operator` モジュールを通じてPython演算子をオーバーロードしている。`x + y` は内部的に `tf.math.add(x, y)` に変換される。

#### Step 2: エントリーポイントを理解する

Python APIのエントリーポイントであるmath_ops.pyの主要関数を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | math_ops.py | `tensorflow/python/ops/math_ops.py` | 数学演算API群 |

**主要処理フロー**:
1. **261行目**: `argmax` 関数 - 最大値インデックス取得
2. **392行目**: `abs` 関数 - 絶対値
3. **473行目**: `divide` 関数 - 除算
4. **508行目**: `multiply` 関数 - 乗算
5. **574行目**: `subtract` 関数 - 減算
6. **700行目**: `pow` 関数 - べき乗
7. **1863行目**: `equal` 関数 - 等値比較
8. **2229行目**: `reduce_sum` 関数 - 総和リダクション
9. **2573行目**: `reduce_mean` 関数 - 平均リダクション
10. **3047行目**: `reduce_max` 関数 - 最大値リダクション
11. **3477行目**: `matmul` 関数 - 行列積
12. **3919行目**: `add` 関数 - 加算
13. **3999行目**: `add_n` 関数 - N個のテンソル加算
14. **4153行目**: `sigmoid` 関数 - シグモイド活性化
15. **4250行目**: `cumsum` 関数 - 累積和

#### Step 3: C++オペレーション定義を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | math_ops.cc | `tensorflow/core/ops/math_ops.cc` | REGISTER_OPマクロによるop定義 |

**主要処理フロー**:
- **33行目**: `AddN` op登録 - N個テンソルの加算、形状推論にMerge使用

### プログラム呼び出し階層図

```
tf.math.add(x, y) [math_ops.py:3919]
    |
    +-- x + y (演算子オーバーロード)
    |
    +-- gen_math_ops.add_v2(x, y)
            |
            +-- C++ AddV2 kernel
                    |
                    +-- BroadcastBinaryOpShapeFn (形状推論)
                    +-- CPU: Eigen実装 / GPU: CUDA実装

tf.math.reduce_sum(input, axis) [math_ops.py:2229]
    |
    +-- gen_math_ops.sum(input, axis)
            |
            +-- C++ Sum kernel
                    +-- ReductionShapeFn (形状推論)
```

### データフロー図

```
[入力]                    [処理]                        [出力]

Tensor x, y       -->  math_ops.py                -->  結果 Tensor
(数値型)                (ブロードキャスト検証)
                            |
                    gen_math_ops.py
                    (C++バインディング)
                            |
                    math_ops.cc
                    (Shape Inference)
                            |
                    C++ Kernel
                    (Eigen/cuBLAS/MKL)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| math_ops.py | `tensorflow/python/ops/math_ops.py` | ソース | Python API公開関数群 |
| math_ops.cc | `tensorflow/core/ops/math_ops.cc` | ソース | C++オペレーション定義 |
| gen_math_ops.py | `tensorflow/python/ops/gen_math_ops.py` | 自動生成 | C++からの自動生成バインディング |
| tensor_math_operator_overrides.py | `tensorflow/python/ops/tensor_math_operator_overrides.py` | ソース | Python演算子オーバーロード |
| override_binary_operator.py | `tensorflow/python/framework/override_binary_operator.py` | ソース | 二項演算子オーバーライド機構 |
| numeric_op.h | `tensorflow/core/framework/numeric_op.h` | ヘッダ | 数値op共通基盤 |
| common_shape_fns.h | `tensorflow/core/framework/common_shape_fns.h` | ヘッダ | 共通形状推論関数 |
